package net.kldp.beat.web.interceptor;

import java.lang.annotation.Annotation;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import net.kldp.beat.action.ActionContext;
import net.kldp.beat.exception.InterceptorException;
import net.kldp.beat.interceptor.Interceptor;
import net.kldp.beat.interceptor.SystemInterceptor;
import net.kldp.beat.interceptor.UserInterceptor;
import net.kldp.beat.web.annotation.ApplicationMap;
import net.kldp.beat.web.annotation.CookieMap;
import net.kldp.beat.web.annotation.HeaderMap;
import net.kldp.beat.web.annotation.Multipart;
import net.kldp.beat.web.annotation.Parameter;
import net.kldp.beat.web.annotation.ParameterMap;
import net.kldp.beat.web.annotation.RequestMap;
import net.kldp.beat.web.annotation.ServletContext;
import net.kldp.beat.web.annotation.ServletDispatcher;
import net.kldp.beat.web.annotation.ServletOutputStream;
import net.kldp.beat.web.annotation.ServletRequest;
import net.kldp.beat.web.annotation.ServletResponse;
import net.kldp.beat.web.annotation.SessionMap;

/**
 * 인터셉터의 인스턴스를 생성하고 리턴하는 팩터리 클래스입니다.
 * 
 */
public abstract class InterceptorFactory implements SystemInterceptor {
	private static HashMap<Class<? extends Annotation>, SystemInterceptor> interceptors = new HashMap<Class<? extends Annotation>, SystemInterceptor>();
	static {
		interceptors.put(ApplicationMap.class, new ApplicationMapInterceptor());
		interceptors.put(Parameter.class, new ParameterInterceptor());
		interceptors.put(ParameterMap.class, new ParameterMapInterceptor());
		interceptors.put(Multipart.class, new MultipartInterceptor());
		interceptors.put(RequestMap.class, new RequestMapInterceptor());
		interceptors.put(ServletContext.class, new ServletContextInterceptor());
		interceptors.put(ServletDispatcher.class, new ServletDispatcherInterceptor());
		interceptors.put(ServletRequest.class, new ServletRequestInterceptor());
		interceptors.put(ServletResponse.class, new ServletResponseInterceptor());
		interceptors.put(ServletOutputStream.class, new ServletOutputStreamInterceptor());
		interceptors.put(HeaderMap.class, new HeaderMapInterceptor());
		interceptors.put(CookieMap.class, new CookieMapInterceptor());
		interceptors.put(SessionMap.class, new SessionMapInterceptor());
	}

	/**
	 * 어노테이션에 맞는 인터셉터를 리턴합니다. 시스템 어노테이션의 경우 싱글턴으로 생성되어 리턴되며, 사용자 인터셉터의 경우 항상 새롭게
	 * 생성되어 리턴됩니다.
	 * 
	 * @param annotation
	 * @return Interceptor
	 * @throws InterceptorException
	 */
	public static Interceptor getInterceptor(Annotation annotation) throws InterceptorException {
		Class<? extends Annotation> clazz = annotation.annotationType();
		if (interceptors.containsKey(clazz))
			return interceptors.get(clazz);
		return createUserInterceptor(clazz);
	}

	public static Interceptor getInterceptor(Class clazz) throws InterceptorException {
		if (interceptors.containsKey(clazz))
			return interceptors.get(clazz);
		return createUserInterceptor(clazz);
	}

	/**
	 * 사용자가 web.interceptor 패키지에 작성한 인터셉터의 인스턴스를 생성하여 리턴합니다.
	 * 
	 * @param clazz
	 * @return Interceptor
	 * @throws InterceptorException
	 */
	private static Interceptor createUserInterceptor(Class<? extends Annotation> clazz)
			throws InterceptorException {
		String interceptorName = "web.interceptor." + clazz.getSimpleName() + "Interceptor";
		try {
			return (Interceptor) Class.forName(interceptorName).newInstance();
		} catch (Exception e) {
			throw new InterceptorException(e);
		}
	}

	/**
	 * 사용자 인터셉터 객체에 시스템 인터셉터를 적용합니다. 시스템 인터셉터가 적용되기 위해서는 사용자 인터셉터가 aware류의
	 * 인터페이스를 상속하거나, 시스템 인터셉터 어노테이션을 선언할 필요가 있습니다.
	 * 
	 * @param target
	 * @param context
	 * @param annotation
	 * @throws InterceptorException
	 */
	public static void intecept(UserInterceptor target, ActionContext context, Annotation annotation)
			throws InterceptorException {
		SystemInterceptor interceptor = interceptors.get(annotation.annotationType());
		if (interceptor != null) {
			interceptor.intercept(target, context, annotation);
		}
	}

	/**
	 * 액션 객체에 인터셉터를 적용합니다. 액션 객체에 인터셉터를 적용하기 위해서는 인터셉터의 어노테이션을 선언하거나, aware류의
	 * 인터페이스를 구현할 필요성이 있습니다.
	 * 
	 * @param action
	 * @param systemInterceptors
	 * @param context
	 * @throws InterceptorException
	 */
	public static void intecept(Object action, Map<SystemInterceptor, Annotation> systemInterceptors,
			ActionContext context) throws InterceptorException {
		List<SystemInterceptor> list = new ArrayList<SystemInterceptor>();
		SystemInterceptor multipartInterceptor = null;
		for (SystemInterceptor interceptor : systemInterceptors.keySet()) {
			if (interceptor instanceof MultipartInterceptor) {
				multipartInterceptor = interceptor;
			} else {
				list.add(interceptor);
			}
		}
		if (multipartInterceptor != null) {
			System.out.println("MULTIPART");
			multipartInterceptor.intercept(action, context, systemInterceptors.get(multipartInterceptor));
		}
		interceptors.get(Parameter.class).intercept(action, context, new Parameter() {
			public Class<? extends Annotation> annotationType() {
				return Parameter.class;
			}
		});
		for (SystemInterceptor interceptor : list) {
			interceptor.intercept(action, context, systemInterceptors.get(interceptor));
		}
	}
}