package net.kldp.beat.action;

import java.io.File;
import java.io.IOException;
import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import net.kldp.beat.annotation.Multipart;
import net.kldp.beat.annotation.Result;
import net.kldp.beat.config.Config;
import net.kldp.beat.exception.InterceptorException;
import net.kldp.beat.exception.ValidationException;
import net.kldp.beat.interceptor.Interceptor;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;

public class ActionService {
	private HttpServletResponse response;
	private HttpServletRequest request;
	private ServletContext context;
	private Object action;

	public ActionService(ServletContext context, HttpServletRequest request, HttpServletResponse response) {
		this.context = context;
		this.request = request;
		this.response = response;
	}

	public void doService(Object action, ActionMapper mapper) throws ServletException {
		this.action = action;
		ServletDispatcher dispatcher = new ServletDispatcherImpl(context, request, response);
		AwareInjector injector = new AwareInjector(context, request, response);
		InterceptorStack stack = new InterceptorStack(action, injector);
		try {
			injectionProperties(stack.getMultipart());
		} catch (FileUploadException e) {
			throw new ServletException("can not upload file" + e);
		}
		injector.injection(action);
		// validation 처리
		try {
			if (!injector.validation(action)) {
				Result input = stack.getResult("input");
				if (input == null)
					throw new ServletException("can not found Result annotation for input.");
				injectionProperties(injector.getRequestMap());
				try {
					dispatcher.dispatch(input, action);
					return;
				} catch (IOException e) {
					throw new ServletException(e);
				}
			}
		} catch (ValidationException e1) {
			throw new ServletException(e1);
		}
		// interceptor 처리
		if (!executeInterceptors(stack.getBeforeInterceptors())) {
			executeInterceptors(stack.getAfterInterceptors());
			return;
		}
		executeMethod(stack.getBeforeMethod());
		Result result;
		try {
			result = stack.getResult(executeAction());
		} catch (InvocationTargetException e) {
			throw new ServletException(e);
		}
		if (result == null) {
			if (stack.getResults().size() == 0)
				result = mapper.getDefaultResult();
			else
				throw new ServletException("can not found Result");
		}
		executeMethod(stack.getBeforeResultMethod());
		injectionProperties(injector.getRequestMap());
		try {
			dispatcher.dispatch(result, action);
		} catch (IOException e) {
			throw new ServletException(e);
		}
		executeMethod(stack.getAfterMethod());
		executeInterceptors(stack.getAfterInterceptors());
	}

	private void executeMethod(Method method) throws ServletException {
		if (method != null) {
			try {
				method.invoke(action, (Object)null);
			} catch (Exception e) {
				throw new ServletException("Can not execute Method");
			}
		}
	}

	private boolean executeInterceptors(Map<Interceptor, Annotation> interceptors) throws ServletException {
		boolean result = true;
		for (Interceptor interceptor : interceptors.keySet()) {
			// Interceptor가 true일경우에만 Action과 View 처리
			try {
				if (!interceptor.intercept(action, interceptors.get(interceptor)))
					result = false;
			} catch (InterceptorException e) {
				throw new ServletException(e);
			}
		}
		return result;
	}

	private void injectionProperties(Multipart multipart) throws FileUploadException {
		try {
			if (multipart != null && ServletFileUpload.isMultipartContent(request)) {
				Map<String, String> parameters = new HashMap<String, String>();
				Map<String, File> files = new HashMap<String, File>();
				DiskFileItemFactory factory = new DiskFileItemFactory();
				// 메모리 버퍼 설정
				factory.setSizeThreshold(1024 * 1024);
				// 임시 저장 디렉토리 설정
				if (!Config.temp_dir().equals(""))
					factory.setRepository(new File(Config.temp_dir()));
				ServletFileUpload upload = new ServletFileUpload(factory);
				// 최대로 저장할 크기 설정
				if (multipart.size() > 0) {
					upload.setSizeMax(multipart.size() * 1024 * 1024);
				} else if (Config.max_upload_size() > 0) {
					upload.setSizeMax(Config.max_upload_size() * 1024 * 1024);
				}
				List items = upload.parseRequest(request);
				Iterator iter = items.iterator();
				while (iter.hasNext()) {
					FileItem item = (FileItem) iter.next();

					if (item.isFormField()) {
						parameters.put(item.getFieldName(), item.getString());
					} else {
						parameters.put(item.getFieldName(), item.getName());
						files.put(item.getFieldName(), new File(item.getName()));
					}
				}
				if (parameters.size() > 0)
					BeanUtils.populate(action, parameters);
				if (files.size() > 0) {
					BeanUtils.populate(action, files);
				}

			} else {
				BeanUtils.populate(action, request.getParameterMap());
			}
		} catch (IllegalAccessException e) {
			// nothing
		} catch (InvocationTargetException e) {
			// nothing
		}
	}

	private void injectionProperties(Map<String, Object> requestMap) {
		try {
			Map<String, Object> temp;
			temp = PropertyUtils.describe(action);
			temp.remove("class");
			requestMap.putAll(temp);
		} catch (IllegalAccessException e) {
			e.printStackTrace();
		} catch (InvocationTargetException e) {
			e.printStackTrace();
		} catch (NoSuchMethodException e) {
			e.printStackTrace();
		}
	}

	private String executeAction() throws InvocationTargetException {
		try {
			Method method = action.getClass().getMethod("execute");
			String result = (String) method.invoke(action);
			return result;
		} catch (Exception e) {
			throw new InvocationTargetException(e);
		}
	}
}